set(DEBIAN_INSTALL_FILE "${CMAKE_BINARY_DIR}/mariadb-plugin-columnstore.install.generated")
file(WRITE ${DEBIAN_INSTALL_FILE} "#File is generated by ColumnstoreLibrary.cmake, do not edit\n")

macro(add_to_debian_install_file file_path)
    string(SUBSTRING "${file_path}" 1 -1 BINARY_ENTRY)
    file(RELATIVE_PATH CMAKEFILE "${CMAKE_SOURCE_DIR}/storage/columnstore/columnstore" ${CMAKE_CURRENT_LIST_FILE})
    string(STRIP "${BINARY_ENTRY}" BINARY_ENTRY)

    file(APPEND ${DEBIAN_INSTALL_FILE} "${BINARY_ENTRY} # added in ${CMAKEFILE}\n")
endmacro()

function(get_target_output_filename TARGET_NAME OUTPUT_VAR)
    # 1. Get the target's OUTPUT_NAME (falls back to TARGET_NAME)
    get_target_property(OUTPUT_NAME ${TARGET_NAME} OUTPUT_NAME)
    if(NOT OUTPUT_NAME)
        set(OUTPUT_NAME "${TARGET_NAME}")
    endif()

    # 1. Get the correct suffix based on target type
    get_target_property(TARGET_TYPE ${TARGET_NAME} TYPE)
    if(TARGET_TYPE STREQUAL "EXECUTABLE")
        set(SUFFIX "${CMAKE_EXECUTABLE_SUFFIX}")
        set(PREFIX "") # No prefix for executables
    elseif(TARGET_TYPE STREQUAL "SHARED_LIBRARY")
        set(SUFFIX "${CMAKE_SHARED_LIBRARY_SUFFIX}")
        set(PREFIX "lib")
    elseif(TARGET_TYPE STREQUAL "STATIC_LIBRARY")
        set(SUFFIX "${CMAKE_STATIC_LIBRARY_SUFFIX}")
        set(PREFIX "lib")
    else()
        message(WARNING "Unknown target type for ${TARGET_NAME}")
        set(SUFFIX "")
    endif()

    # 1. Combine into the final filename
    set(${OUTPUT_VAR}
        "${PREFIX}${OUTPUT_NAME}${SUFFIX}"
        PARENT_SCOPE
    )
endfunction()

macro(columnstore_install_target target destination)
    install(
        TARGETS ${target}
        DESTINATION ${destination}
        COMPONENT columnstore-engine
    )
    get_target_output_filename(${target} OUTPUT_FILENAME)
    add_to_debian_install_file("${destination}/${OUTPUT_FILENAME}")
endmacro()

macro(columnstore_install_file file destination)
    install(
        FILES ${file}
        DESTINATION ${destination}
        COMPONENT columnstore-engine
    )
    get_filename_component(FILENAME ${file} NAME)
    add_to_debian_install_file("${destination}/${FILENAME}")
endmacro()

macro(columnstore_install_program file destination)
    install(
        PROGRAMS ${file}
        DESTINATION ${destination}
        COMPONENT columnstore-engine
    )
    get_filename_component(FILENAME ${file} NAME)
    add_to_debian_install_file(
        "${destination}/${FILENAME}
        "
    )
endmacro()

macro(columnstore_static_library libname)
    add_definitions(-fPIC -DPIC)
    add_library(${libname} STATIC ${ARGN})
endmacro()

macro(columnstore_shared_library libname)
    add_library(${libname} SHARED ${ARGN})
    columnstore_install_target(${libname} ${ENGINE_LIBDIR})
endmacro()

macro(columnstore_library libname)
    if(COLUMNSTORE_STATIC_LIBRARIES)
        columnstore_static_library(${libname} ${ARGN})
    else()
        columnstore_shared_library(${libname} ${ARGN})
    endif()
endmacro()

macro(columnstore_mysql_plugin_library libname)
    add_library(${libname} SHARED ${ARGN})
    columnstore_install_target(${libname} ${MARIADB_PLUGINDIR})
endmacro()

macro(columnstore_link libname)
    target_link_libraries(${libname} ${ARGN})
endmacro()

macro(columnstore_executable executable_name)
    if(WITH_COLUMNSTORE_ASAN)
        add_executable(${executable_name} ${ARGN} ${CMAKE_BINARY_DIR}/asan_options.cpp)
    else()
        add_executable(${executable_name} ${ARGN})
    endif()
    columnstore_install_target(${executable_name} ${ENGINE_BINDIR})
endmacro()

# Read /etc/os-release and output: ID (lowercase) and VERSION_ID major number
function(columnstore_detect_os OUT_ID OUT_VER_MAJOR)
    set(_os_id "")
    set(_os_version_major "")

    set(_os_release "/etc/os-release")
    if(EXISTS "${_os_release}")
        file(READ "${_os_release}" _osr)
        # Extract ID
        string(REGEX MATCH "\nID=([^\n]+)" _id_match "\nID=([^\n]+)" ${_osr})
        if(_id_match)
            string(REGEX REPLACE ".*\nID=\"?([^\"\n]+)\"?.*" "\\1" _os_id "${_osr}")
            string(TOLOWER "${_os_id}" _os_id)
        endif()
        # Extract VERSION_ID major digits
        string(REGEX MATCH "\nVERSION_ID=([^\n]+)" _vid_match "\nVERSION_ID=([^\n]+)" ${_osr})
        if(_vid_match)
            string(REGEX REPLACE ".*\nVERSION_ID=\"?([0-9]+).*" "\\1" _os_version_major "${_osr}")
        endif()
    endif()

    set(${OUT_ID}
        "${_os_id}"
        PARENT_SCOPE
    )
    set(${OUT_VER_MAJOR}
        "${_os_version_major}"
        PARENT_SCOPE
    )
endfunction()

# Check whether a given lowercase OS ID is RHEL-like (RHEL/Rocky/Alma/CentOS/RedHat)
function(columnstore_is_rhel_like OS_ID OUT_BOOL)
    set(_is_rhel_like FALSE)
    if(${OS_ID} MATCHES "^(rhel|rocky|almalinux|centos|redhatenterpriseserver|redhatenterprise|redhat)$")
        set(_is_rhel_like TRUE)
    endif()
    set(${OUT_BOOL}
        "${_is_rhel_like}"
        PARENT_SCOPE
    )
endfunction()
